import SwiftUI

struct PaymentEngineView: View {
    @State private var deviceName: String = "Verifone_P400.4"
    @State private var amount: String = ""
    @State private var ip: String = "172.21.2.6"
    @State private var port: String = "9001"
    @State private var timeout: String = "";
    @State private var output: String = ""
    
    @EnvironmentObject var cardknoxSDKManager: CardknoxSDKManager
    
    let transactionResultSubscription =
    NotificationCenter.default.publisher(for:
    NSNotification.Name(CardknoxSDK.transactionResultSubscription_NSNotificationCenterName()))
    
    let deviceStatusSubscription =
    NotificationCenter.default.publisher(for:
    NSNotification.Name(CardknoxPaymentEngineSDK.deviceStatus_NSNotification_Name()))
    
    let transactionStatusSubscription =
    NotificationCenter.default.publisher(for:
    NSNotification.Name(CardknoxPaymentEngineSDK.transactionStatus_NSNotification_Name()))
    
    
    private var twoColumnGrid = [GridItem(.flexible()), GridItem(.flexible())]
    
    var body: some View {
        
        GeometryReader { geometry in
            ScrollView(.vertical){
                VStack(spacing: 21){
                    VStack(spacing: 15){
                        Text("Payment Engine")
                            .font(Font.custom("SFProDisplay-Semibold", size: 18))
                            .frame(maxWidth: .infinity, alignment: .leading)
                        Divider()
                            .frame(maxWidth: .infinity, maxHeight: 1)
                            .background(Color("Gray1"))
                    }
                
                    VStack{
                        LazyVGrid(columns: twoColumnGrid, alignment: .leading){
                            // Display the item
                            Text("Device Name")
                            TextField("Enter device name", text: $deviceName)
                                .textFieldStyle(RoundedBorderTextFieldStyle())
                            
                            Text("Amount")
                            TextField("Enter amount", text: $amount)
                                .textFieldStyle(RoundedBorderTextFieldStyle())
                                .keyboardType(UIKeyboardType.decimalPad)
                            
                            Text("IP")
                            TextField("Enter IP", text: $ip)
                                .textFieldStyle(RoundedBorderTextFieldStyle())
                                                            
                            Text("Port")
                            TextField("Enter port", text: $port)
                                .textFieldStyle(RoundedBorderTextFieldStyle())
                            
                            Text("Timeout")
                            TextField("Default 120s", text: $timeout)
                                .textFieldStyle(RoundedBorderTextFieldStyle())
                                .keyboardType(UIKeyboardType.decimalPad)
                            
                        }
                        
                        Button(action: executeDeviceInitialize){
                            Text("Device_Initialize")
                        }
                        .buttonStyle(RedButton())
                        
                        Button(action: executeProcessOutOfScope){
                            Text("ProcessOutOfScope")
                        }
                        .buttonStyle(RedButton())
                        
                        Button(action: executeCancelTransaction){
                            Text("Device_CancelTransaction")
                        }
                        .buttonStyle(RedButton())
                        
                        VStack(spacing: 15){
                            LazyVGrid(columns: twoColumnGrid, alignment: .leading){
                                Text("Output")
                                    .font(Font.custom("SFProDisplay-Semibold", size: 18))
                                    .frame(maxWidth: .infinity, alignment: .leading)
                                
                                Button(action: executeClearOutput){
                                    Text("Clear")
                                }
                                .buttonStyle(WhiteButton())
                            }
                            
                            Divider()
                                .frame(maxWidth: .infinity, maxHeight: 1)
                                .background(Color("Gray1"))
                        }
                        
                        VStack {
                            ScrollView(.vertical, showsIndicators: false){
                                Text(output)
                                    .lineLimit(nil)
                                    .foregroundColor(Color("Gray3"))
                                    .background(Color.white)
                                    .frame(maxWidth: .infinity, maxHeight: .infinity, alignment: Alignment.topLeading)
                            }
                            .frame(idealHeight: 325, maxHeight: .infinity)
                        }
                        .padding(11)
                        .frame(maxWidth: .infinity, maxHeight: .infinity, alignment: Alignment.topLeading)
                        .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                    }
                }
                .padding(20)
            }
            .background(Color.white)
        }
        .onAppear(perform: viewDidAppear)
        .onTapGesture { self.hideKeyboard() }
        .onReceive(transactionResultSubscription, perform: transactionResultNotification(aNotification:))
        .onReceive(transactionStatusSubscription, perform: transactionStatusNotification(aNotification:))
        .onReceive(deviceStatusSubscription, perform: deviceStatusNotification(aNotification:))
    }
    
    func transactionStatusNotification(aNotification: Notification){
        let status = CardknoxPaymentEngineSDK.unwrapTransactionStatus(with: aNotification) as String;
        let line = "transaction status: " + status;
        addLineToOutput(line: line)
    }
    
    func deviceStatusNotification(aNotification: Notification){
        let status = CardknoxPaymentEngineSDK.unwrapDeviceStatus(with: aNotification) as String;
        let line = "device status: " + status;
        addLineToOutput(line: line)
    }
    
    func transactionResultNotification(aNotification: Notification)
    {
        let response = PaymentTransactionResponse.unwrap(aNotification) as! PaymentTransactionResponse
        
        var message = ""
        if(response.isSuccess())
        {
            let refNum = response.xRefNum()!
            message = "Transaction success response! Ref num: " + refNum
        }
        else
        {
            let error : String? = response.errorMessage();
            if(error != nil){
                message = "Transaction error response - " + error!
            }
         
        }
        addLineToOutput(line: message)
    }
    
    func viewDidAppear()
    {
        // RegionStart - InitializeSDK
        CardknoxSDK.setPrincipalKey("Your xKey")
        CardknoxSDK.setxSoftwareName("Your app name", xVersion: "4.5.9")
        // RegionEnd - InitializeSDK
        
        CardknoxSDK.setPrincipalKey("ps_demo_cardknox")
        CardknoxSDK.setxSoftwareName("Your app name", xVersion: "4.5.9")
        
             
        // Documentation
        if(true){
            // This will output all PaymentEngine logger outputs to the Console
            // To disable; pass in "false"
            cardknoxSDKManager.getCardknoxSDKPaymentEngine().enableConsoleOutput(true);
        }
    }
    
    func executeClearOutput(){
        self.output = "";
    }
    
    func executeDeviceInitialize(){
        self.hideKeyboard()
        
        let paymentEngine = cardknoxSDKManager.getCardknoxSDKPaymentEngine()!;
        let settings = getSettings()
        paymentEngine.device_Initialize(withSettings: settings)
        
    }

    func executeProcessOutOfScope(){
        self.hideKeyboard()
        let paymentEngine = cardknoxSDKManager.getCardknoxSDKPaymentEngine()!;
        
        let settings = getSettings()
        let request = getRequest()
        CardknoxSDK.enableLogging(true)
        paymentEngine.processOutOfScope(withSettings: settings, request: request)
    }
    
    func executeCancelTransaction(){
        let paymentEngine = cardknoxSDKManager.getCardknoxSDKPaymentEngine()!;
        paymentEngine.device_CancelTransaction();
    }
    
    func getSettings() -> CardknoxPaymentEngineSDKSettings{
        let settings : CardknoxPaymentEngineSDKSettings = CardknoxPaymentEngineSDKSettings.init();
        settings.device_Name = deviceName as NSString?;
        settings.device_IP_Port = port as NSString?;
        
        if(timeout != nil && !timeout.isEmpty) {
            settings.device_Timeout = Int32(timeout)! * 1000; // value * milliseconds = seconds
        }
        else{
            settings.device_Timeout = 120 * 1000; // seconds
        }
        settings.device_IP_Address = ip as NSString?
        return settings;
    }

    func getRequest() -> CardknoxPaymentEngineSDKRequest{
        let request : CardknoxPaymentEngineSDKRequest = CardknoxPaymentEngineSDKRequest.init();
        request.xAllowDuplicate = true;
        if(amount != "" && amount != nil) {
                 request.xAmount = Double(amount)!
             }
             else{
                 addLineToOutput(line: "Missing amount")
             }
        request.xCommand = "cc:sale";
        return request;
    }
    
    func addLineToOutput(line : String)
    {
        NSLog("PaymentEngine: %@", line);
        DispatchQueue.main.async
        {
            output = output + line + "\n"
        }
    }
}
       




