import Foundation
import SwiftUI

struct OutOfScopeView : View
{
    @State private var commandType: String? = ""
    @State private var amount: String = ""
    @State private var output: String = ""
    @State private var showKeyedScreen = true
    @State private var showSwipeScreen = true
    @State private var closeUIOnProcessedTransaction = true
    @EnvironmentObject var cardknoxSDKManager: CardknoxSDKManager
    
    // RegionStart PublisherTransaction
    let transactionResultSubscriptionPublisher =
    NotificationCenter.default.publisher(for: NSNotification.Name(CardknoxSDK.transactionResultSubscription_NSNotificationCenterName()))
    // RegionEnd PublisherTransaction
    
    // RegionStart PublisherCardReader
    let cardreaderEventSubscriptionPublisher =
    NotificationCenter.default.publisher(for: NSNotification.Name(CardknoxSDK.cardreaderEventSubscription_NSNotificationCenterName()))
    // RegionEnd PublisherCardReader
    
    init()
    {
    }
    
    var body: some View
    {
        GeometryReader { geometry in
            ScrollView(.vertical)
            {
                VStack(spacing: 21){
                    VStack(spacing: 15){
                        Text("Process out of scope")
                            .font(Font.custom("SFProDisplay-Semibold", size: 18))
                            .frame(maxWidth: .infinity, alignment: .leading)
                        Divider()
                            .frame(maxWidth: .infinity, maxHeight: 1)
                            .background(Color("Gray1"))
                    }
                    HStack
                    {
                        CommandTypePickerView(text: $commandType)
                        Image("ArrowDown_24")
                            .frame(width: 24, height: 24, alignment: Alignment.center)
                    }
                    .padding()
                    .frame(height: 44)
                    .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                    
                    Toggle(isOn: $showKeyedScreen)
                    {
                        Text("Show keyed screen")
                            .font(Font.custom("SFProDisplay-Regular", size: 16))
                    }
                    
                    Toggle(isOn: $showSwipeScreen)
                    {
                        Text("Show swipe screen")
                            .font(Font.custom("SFProDisplay-Regular", size: 16))
                    }
                    
                    Toggle(isOn: $closeUIOnProcessedTransaction)
                    {
                        Text("Close UI on processed transaction")
                            .font(Font.custom("SFProDisplay-Regular", size: 16))
                    }
                    
                    TextField("Amount", text: $amount)
                        .textFieldStyle(RoundedTextFieldWithBorderStyle())
                        .keyboardType(UIKeyboardType.decimalPad)
                    
                    HStack(alignment: VerticalAlignment.firstTextBaseline)
                    {
                        Button(action: showCardknoxUIButtonClick)
                        {
                            Text("Show Cardknox UI")
                        }
                        .buttonStyle(RedButton())
                        
                        Spacer()
                            .frame(maxWidth: .infinity)
                        
                    }
                    
                    VStack(spacing: 15){
                        Text("Output")
                            .font(Font.custom("SFProDisplay-Semibold", size: 18))
                            .frame(maxWidth: .infinity, alignment: .leading)
                        Divider()
                            .frame(maxWidth: .infinity, maxHeight: 1)
                            .background(Color("Gray1"))
                    }
                    VStack {
                        ScrollView(.vertical, showsIndicators: false){
                            Text(output)
                                .lineLimit(nil)
                                .foregroundColor(Color("Gray3"))
                                .background(Color.white)
                                .frame(maxWidth: .infinity, maxHeight: .infinity, alignment: Alignment.topLeading)
                        }
                    }
                    .padding(11)
                    .frame(maxWidth: .infinity, maxHeight: .infinity, alignment: Alignment.topLeading)
                    .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                }
                .padding(20)
                .frame(height: geometry.size.height)
                
            }
            .background(Color.white)
        }
        .onAppear(perform: viewDidAppear)
        .onTapGesture { self.hideKeyboard() }
        .onReceive(transactionResultSubscriptionPublisher, perform:  transactionResultSubscription(aNotification:))
        .onReceive(cardreaderEventSubscriptionPublisher, perform: cardreaderEventSubscription(aNotification:))
    }

    
    func viewDidAppear()
    {
        // RegionStart - InitializeSDK
        CardknoxSDK.setPrincipalKey("Your xKey")
        CardknoxSDK.setxSoftwareName("Your app name", xVersion: "4.5.9")
        // RegionEnd - InitializeSDK
        
        CardknoxSDK.setPrincipalKey("ps_demo_cardknox")
        CardknoxSDK.setxSoftwareName("Your app name", xVersion: "4.5.9")
    }
    
    func showCardknoxUIButtonClick()
    {
        self.hideKeyboard()
        output = ""
        let cardknoxSDKUI = cardknoxSDKManager.getCardknoxSDKUI()
        
        // Documentation
        if(false){
            // Define if a 'keyed' screen is available
            cardknoxSDKManager.getCardknoxSDKUI().enableKeyedEntry = true;
            // Define if a 'swipe' screen is available
            cardknoxSDKManager.getCardknoxSDKUI().enableDeviceInsertSwipeTap = true;
            // Define if the UI should auto close
            cardknoxSDKManager.getCardknoxSDKUI().closeSDKUIOnProcessedTransaction = true;
        }
        
        // Define if a 'keyed' screen is available
        cardknoxSDKManager.getCardknoxSDKUI().enableKeyedEntry = showKeyedScreen;
        // Define if a 'swipe' screen is available
        cardknoxSDKManager.getCardknoxSDKUI().enableDeviceInsertSwipeTap = showSwipeScreen;
        // Define if the UI should auto close
        cardknoxSDKManager.getCardknoxSDKUI().closeSDKUIOnProcessedTransaction = closeUIOnProcessedTransaction;
       
        // RegionStart - CreateParams
        let prms : TransactionParameters = TransactionParameters.init()
        // RegionEnd - CreateParams
        
        // RegionStart - SetRequiredParams
        prms.xCommand = "<xCommand>";
        prms.xAmount = 1.23;
        // RegionEnd - SetRequiredParams
        
        // RegionStart - SetOptionalParams
        prms.xInvoice = "123456";
        prms.xBillFirstName = "Billing first name";
        // ... etc.
        // RegionEnd - SetOptionalParams
        
        prms.xCommand = commandType as NSString?
        
        let amountDouble = Double( amount.replacingOccurrences(of: ",", with: ".")) ?? 0
        prms.xAmount = amountDouble
        
        // RegionStart - CreateRequest
        let request = cardknoxSDKUI?.createRequest(withParameters: prms) as! PaymentTransactionRequestUI
        // RegionEnd - CreateRequest
        
        
        prms.vp3300TransactionTimeout = 12;
        
        // Documentation - vp3300 timeout
        if(false){
            let ui = (CardknoxSDK.create() as! CardknoxSDK).getOutOfScope() as! CardknoxSDKUI;
            let parameters : TransactionParameters = TransactionParameters.init()
            parameters.vp3300TransactionTimeout = 13;
            let req = cardknoxSDKUI?.createRequest(withParameters: parameters) as! PaymentTransactionRequestUI
            // Enable swipe screen in the Cardknox UI
            ui.enableDeviceInsertSwipeTap = true;
            // Show Cardknox UI
            req.process();
        }
        
        if(request.isValid)
        {
            // RegionStart - CallProcess
            // Show the Cardknox UI
            request.process()
            // RegionEnd - CallProcess
        }
        else
        {
            // RegionStart - ReadErrors
            // Request could not be processed due to these errors
            let errors = request.validationErrors!
            // RegionEnd - ReadErrors
            
            var outputText = ""
            for item in errors
            {
                outputText += item as! String + "\n"
            }
            output = outputText
        }
    }
    
    // RegionStart - SubscribeTransactionMethod
    func transactionResultSubscription(aNotification: Notification)
    {
        let response = PaymentTransactionResponse.unwrap(aNotification) as! PaymentTransactionResponse
        
        var message = ""
        if(response.isSuccess())
        {
            let refNum = response.xRefNum()!
            message = "Transaction success response! Ref num: " + refNum
        }
        else
        {
            let error = response.errorMessage()!
            message = "Transaction error response - " + error
        }
        // RegionEnd - SubscribeTransactionMethod
        
        addLineToOutput(line: message)
        
        // RegionStart - SubscribeTransactionMethod
    }
    // RegionEnd - SubscribeTransactionMethod
    
    // RegionStart - SubscribeCardReaderMethod
    func cardreaderEventSubscription(aNotification: Notification)
    {
        let callback = CardknoxCardReaderCallback.unwrap(aNotification) as? CardknoxCardReaderCallback
        
        let code = callback?.code()
        let name = callback?.name()
        
        var errorMessage : String?;
        if(code == CardknoxCardReaderCallbackType.error()){
            errorMessage = callback?.message();
        }
        // RegionEnd - SubscribeCardReaderMethod

        addLineToOutput(line: "Card reader - " + name!)
        if(errorMessage != nil){
            addLineToOutput(line: "Card reader - " + errorMessage!)
        }
        
        // RegionStart - SubscribeCardReaderMethod
    }
    // RegionEnd - SubscribeCardReaderMethod
    
    func addLineToOutput(line : String)
    {
        NSLog("----- TEST ----- : %@", line);
        DispatchQueue.main.async
        {
            output = output + line + "\n"
        }
    }
}

