import Foundation
import SwiftUI

struct InScopeView : View
{
    @State private var commandType: String? = ""
    @State private var amount: String = ""
    @State private var cardNumber: String = ""
    @State private var xtoken: String = ""
    @State private var expDate: String? = ""
    @State private var refNumber: String = ""
    @State private var output: String = ""
    @State var showAdvancedOptions: Bool = false
    @EnvironmentObject var advancedOptions: AdvancedOptions
    @EnvironmentObject var cardknoxSDKManager: CardknoxSDKManager
    
    init()
    {
    }
    
    var body: some View
    {
        // Using NavigationView as a View example
        NavigationView{
            GeometryReader { geometry in
                ScrollView(.vertical)
                {
                    VStack(spacing: 21){
                        VStack(spacing: 15){
                            Text("Process in scope")
                                .font(Font.custom("SFProDisplay-Semibold", size: 18))
                                .frame(maxWidth: .infinity, alignment: .leading)
                            Divider()
                                .frame(maxWidth: .infinity, maxHeight: 1)
                                .background(Color("Gray1"))
                        }
                        HStack
                        {
                            CommandTypePickerView(text: $commandType)
                            Image("ArrowDown_24")
                                .frame(width: 24, height: 24, alignment: Alignment.center)
                        }
                        .padding()
                        .frame(height: 44)
                        .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                        
                        TextField("Amount", text: $amount)
                            .textFieldStyle(RoundedTextFieldWithBorderStyle())
                            .keyboardType(UIKeyboardType.decimalPad)
                        
                        TextField("Token", text: $xtoken)
                            .textFieldStyle(RoundedTextFieldWithBorderStyle())
                            .keyboardType(UIKeyboardType.default)
                        
                        TextField("Card number", text: $cardNumber)
                            .textFieldStyle(RoundedTextFieldWithBorderStyle())
                            .keyboardType(UIKeyboardType.numberPad)
                            .visibility(!Commands.isRefNum(command: commandType!))
                        
                        HStack
                        {
                            ExpirationDatePickerView(text: $expDate)
                            Image("ArrowDown_24")
                                .frame(width: 24, height: 24, alignment: Alignment.center)
                        }
                        .padding()
                        .frame(height: 44)
                        .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                        .visibility(!Commands.isRefNum(command: commandType!))
                        
                        TextField("Reference number", text: $refNumber)
                            .textFieldStyle(RoundedTextFieldWithBorderStyle())
                            .keyboardType(UIKeyboardType.numberPad)
                            .visibility(Commands.isRefNum(command: commandType!))
                        
                        HStack(spacing: 21)
                        {
                            Button(action: executeRequestButtonClick)
                            {
                                Text("Execute request")
                            }
                            .buttonStyle(RedButton())
                            
                            NavigationLink(destination: AdvancedOptionsView(showSelf: $showAdvancedOptions), isActive: $showAdvancedOptions){
                                Button("More options")
                                {
                                    self.showAdvancedOptions = true
                                }
                                .buttonStyle(WhiteButton())
                            }
                        }
                        
                        VStack(spacing: 15){
                            Text("Output")
                                .font(Font.custom("SFProDisplay-Semibold", size: 18))
                                .frame(maxWidth: .infinity, alignment: .leading)
                            Divider()
                                .frame(maxWidth: .infinity, maxHeight: 1)
                                .background(Color("Gray1"))
                        }
                        VStack {
                            ScrollView(.vertical, showsIndicators: false){
                                Text(output)
                                    .lineLimit(nil)
                                    .foregroundColor(Color("Gray3"))
                                    .background(Color.white)
                                    .frame(maxWidth: .infinity, maxHeight: .infinity, alignment: Alignment.topLeading)
                            }
                        }
                        .padding(11)
                        .frame(maxWidth: .infinity, maxHeight: .infinity, alignment: Alignment.topLeading)
                        .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                    }
                    .padding(20)
                    .frame(height: geometry.size.height)
                    
                }
                .background(Color.white)
            }
            .navigationBarHidden(true)
        }
        .onAppear(perform: viewDidAppear)
        .onTapGesture { self.hideKeyboard() }
    }

   
    func viewDidAppear()
    {
        // RegionStart - InitializeSDK
        CardknoxSDK.setPrincipalKey("Your xKey")
        CardknoxSDK.setxSoftwareName("Your app name", xVersion: "4.5.9")
        // RegionEnd - InitializeSDK
        
        CardknoxSDK.setPrincipalKey("ps_demo_cardknox")
        CardknoxSDK.setxSoftwareName("Your app name", xVersion: "4.5.9")
    }
    
    func executeRequestButtonClick()
    {
        self.hideKeyboard()
        output = ""
        
        // RegionStart - CreateParams
        let prms : TransactionParameters = TransactionParameters.init()
        // RegionEnd - CreateParams
        
        if(false)
        {
            // RegionStart - SetRequiredParams
            prms.xCommand = "<xCommand>";
            prms.xAmount = 1.23;
            // RegionEnd - SetRequiredParams
            
            // RegionStart - SetRequiredParams
            prms.xCommand = "<xCommand>";
            prms.xAmount = 1.23;
            // RegionEnd - SetRequiredParams
            
            // RegionStart - SetAdditionalParams
            // Ref num transaction types
            prms.xRefNum = "123456789";
            
            // Card number transaction types
            prms.xCardNum = "4444333322221111";
            prms.xExp = "1225";
            // RegionEnd - SetAdditionalParams
            
            // RegionStart - SetOptionalParams
            prms.xInvoice = "123456";
            prms.xBillFirstName = "Billing first name";
            // ... etc.
            // RegionEnd - SetOptionalParams
        }
        
        let command = commandType as NSString?
        prms.xCommand = command
        
        let amountDouble = Double( amount.replacingOccurrences(of: ",", with: ".")) ?? 0
        prms.xAmount = amountDouble
        
        
        if(Commands.isRefNum(command: command! as String))
        {
            prms.xRefNum = refNumber as NSString?
        }
        else
        {
            prms.xToken = xtoken as NSString?
            
            prms.xCardNum = cardNumber as NSString?
            
            let exp = expDate?.replacingOccurrences(of: "|", with: "")
            prms.xExp = exp as NSString?
        }
        
        prms.xInvoice = advancedOptions.Invoice as NSString?
        prms.xPoNum = advancedOptions.PONumber as NSString?
        prms.xDescription = advancedOptions.Description as NSString?
        
        prms.xBillFirstName = advancedOptions.BillFirstName as NSString?
        prms.xBillLastName = advancedOptions.BillLastName as NSString?
        prms.xBillCompany = advancedOptions.BillCompany as NSString?
        prms.xBillStreet = advancedOptions.BillAddress as NSString?
        prms.xBillCity = advancedOptions.BillCity as NSString?
        prms.xBillZip = advancedOptions.BillZip as NSString?
        prms.xBillCountry = advancedOptions.BillCountry as NSString?
        prms.xBillPhone = advancedOptions.BillPhoneNumber as NSString?
        
        prms.xShipFirstName = advancedOptions.ShipFirstName as NSString?
        prms.xShipLastName = advancedOptions.ShipLastName as NSString?
        prms.xShipCompany = advancedOptions.ShipCompany as NSString?
        prms.xShipStreet = advancedOptions.ShipAddress as NSString?
        prms.xShipCity = advancedOptions.ShipCity as NSString?
        prms.xShipZip = advancedOptions.ShipZip as NSString?
        prms.xShipCountry = advancedOptions.ShipCountry as NSString?
        prms.xShipPhone = advancedOptions.ShipPhoneNumber as NSString?
        
        let cardknoxDirect = cardknoxSDKManager.getCardknoxSDKDirect()!;
        
        // RegionStart - CreateRequest
        let request = cardknoxDirect.createRequest(withParameters: prms) as! PaymentTransactionRequestDirect
        // RegionEnd - CreateRequest
        
        var outputText = ""
        if(request.isValid)
        {
            // RegionStart - CallProcess
            let response = request.process() as! PaymentTransactionResponse
            // RegionEnd - CallProcess
            
            if(response.isSuccess())
            {
                // RegionStart - ReadSuccess
                let refNum = response.xRefNum();
                let status = response.xStatus();
                let avs = response.xAvsResult();
                // RegionEnd - ReadSuccess
                outputText = "Transaction successfully created."
            }
            else
            {
                // RegionStart - ReadFailure
                let refNum = response.xRefNum();
                var error :String? = nil;
                if(response != nil && response.xError() != nil){
                    error = response.xError()
                }
                    
                // RegionEnd - ReadFailure
                outputText += "Transaction failed\n"
                if(response.errorMessage() != nil)
                {
                    outputText += "Error message: " + response.errorMessage() + "\n"
                }
                if(response.xError() != nil)
                {
                    outputText += "Error: " + response.xError() + "\n"
                }
                if(response.xErrorCode() != nil)
                {
                    outputText += "Error code: " + response.xErrorCode() + "\n"
                }
                if(response.xRefNum() != nil)
                {
                    outputText += "Ref num: " + response.xRefNum() + "\n"
                }
            }
        }
        else
        {
            // RegionStart - ReadErrors
            // Request could not be processed due to these errors
            let errors = request.validationErrors!
            // RegionEnd - ReadErrors
            for item in errors
            {
                outputText += item as! String + "\n"
            }
        }
        
        NSLog("----- TEST ----- : %@", outputText);
        output = outputText
    }
}
