import Foundation
import SwiftUI

struct CustomUIView : View
{
    @State private var scanningTimeout: String? = ""
    @State private var scannedDevices: [CardknoxSDKCustomUIBLEScannedDevice]! = [CardknoxSDKCustomUIBLEScannedDevice]()
    @State private var scannedDevice: CardknoxSDKCustomUIBLEScannedDevice!
    @State private var commandType: String? = ""
    @State private var amount: String = ""
    @State private var output: String = ""
    @EnvironmentObject var cardknoxSDKManager: CardknoxSDKManager
    
    let transactionResultSubscription =
    NotificationCenter.default.publisher(for: NSNotification.Name(CardknoxSDK.transactionResultSubscription_NSNotificationCenterName()))
    
    let cardreaderEventSubscription =
    NotificationCenter.default.publisher(for: NSNotification.Name(CardknoxSDK.cardreaderEventSubscription_NSNotificationCenterName()))
    
    let customUIscanCompletedSubscription =
    NotificationCenter.default.publisher(for: NSNotification.Name(CardknoxSDKCustomUIBLE.customUI_scanCompleted_Subscription_NSNotificationCenterName()))
    
    let customUIscannedDeviceSubscription =
    NotificationCenter.default.publisher(for: NSNotification.Name(CardknoxSDKCustomUIBLE.customUI_scannedDevice_Subscription_NSNotificationCenterName()))
    
    init()
    {
    }
    
    var body: some View
    {
        GeometryReader{ geometry in
            ScrollView(.vertical)
            {
                VStack(spacing: 21){
                    VStack(spacing: 15){
                        Text("Custom UI integration")
                            .font(Font.custom("SFProDisplay-Semibold", size: 18))
                            .frame(maxWidth: .infinity, alignment: .leading)
                        Divider()
                            .frame(maxWidth: .infinity, maxHeight: 1)
                            .background(Color("Gray1"))
                    }
                    
                    HStack
                    {
                        ScanningTimeoutPickerView(text: $scanningTimeout)
                        Image("ArrowDown_24")
                            .frame(width: 24, height: 24, alignment: Alignment.center)
                    }
                    .padding()
                    .frame(height: 44)
                    .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                    
                    HStack(alignment: VerticalAlignment.firstTextBaseline)
                    {
                        Button(action: startScanningButtinClick)
                        {
                            Text("Start scanning")
                        }
                        .buttonStyle(RedButton())
                        
                        Button(action: stopScanningButtinClick)
                        {
                            Text("Stop scanning")
                        }
                        .buttonStyle(RedButton())
                        
                    }
                    
                    HStack
                    {
                        ScannedDevicePickerView(selectDevice: $scannedDevice, dataSource: $scannedDevices)
                        Image("ArrowDown_24")
                            .frame(width: 24, height: 24, alignment: Alignment.center)
                    }
                    .padding()
                    .frame(height: 44)
                    .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                    
                    HStack(alignment: VerticalAlignment.firstTextBaseline)
                    {
                        Button(action: connectButtinClick)
                        {
                            Text("Connect")
                        }
                        .buttonStyle(RedButton())
                        
                        Button(action: disconnectButtinClick)
                        {
                            Text("Disconnect")
                        }
                        .buttonStyle(RedButton())
                        
                    }
                    
                    HStack
                    {
                        CommandTypePickerView(text: $commandType)
                        Image("ArrowDown_24")
                            .frame(width: 24, height: 24, alignment: Alignment.center)
                    }
                    .padding()
                    .frame(height: 44)
                    .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                    
                    TextField("Amount", text: $amount)
                        .textFieldStyle(RoundedTextFieldWithBorderStyle())
                        .keyboardType(UIKeyboardType.decimalPad)
                    
                    HStack(alignment: VerticalAlignment.firstTextBaseline)
                    {
                        Button(action: startTransactionButtonClick)
                        {
                            Text("Start transaction")
                        }
                        .buttonStyle(RedButton())
                        
                        Button(action: stopTransactionButtinClick)
                        {
                            Text("Stop transaction")
                        }
                        .buttonStyle(RedButton())
                        
                    }
                    
                    VStack(spacing: 15){
                        Text("Output")
                            .font(Font.custom("SFProDisplay-Semibold", size: 18))
                            .frame(maxWidth: .infinity, alignment: .leading)
                        Divider()
                            .frame(maxWidth: .infinity, maxHeight: 1)
                            .background(Color("Gray1"))
                    }
                    VStack {
                        ScrollView(.vertical, showsIndicators: false){
                            Text(output)
                                .lineLimit(nil)
                                .foregroundColor(Color("Gray3"))
                                .background(Color.white)
                                .frame(maxWidth: .infinity, maxHeight: .infinity, alignment: Alignment.topLeading)
                        }
                        .frame(idealHeight: 325, maxHeight: .infinity)
                    }
                    .padding(11)
                    .frame(maxWidth: .infinity, maxHeight: .infinity, alignment: Alignment.topLeading)
                    .background(RoundedRectangle(cornerRadius: 6).strokeBorder(Color("Gray2"), style: StrokeStyle(lineWidth: 1)))
                }
                .padding(20)
                
            }
            .background(Color.white)
        }
        .onAppear(perform: viewDidAppear)
        .onTapGesture { self.hideKeyboard()}
        .onReceive(transactionResultSubscription, perform:  transactionResultNotification(aNotification:))
        .onReceive(cardreaderEventSubscription, perform: cardreaderEventNotification(aNotification:))
        .onReceive(customUIscanCompletedSubscription, perform: customUIscanCompletedNotification(aNotification:))
        .onReceive(customUIscannedDeviceSubscription, perform: customUIscannedDeviceSubscription)
    }
    
    func viewDidAppear()
    {
        CardknoxSDK.setPrincipalKey("ps_demo_cardknox")
        CardknoxSDK.setxSoftwareName("Your app name", xVersion: "4.5.9")
    }
    
    func transactionResultNotification(aNotification: Notification)
    {
        let response = PaymentTransactionResponse.unwrap(aNotification) as! PaymentTransactionResponse
        
        var message = ""
        if(response.isSuccess())
        {
            let refNum = response.xRefNum()!
            message = "Transaction success response! Ref num: " + refNum
        }
        else
        {
            let error = response.errorMessage()!
            message = "Transaction error response - " + error
        }
        addLineToOutput(line: message)
    }
    
    func cardreaderEventNotification(aNotification: Notification)
    {
        let callback = CardknoxCardReaderCallback.unwrap(aNotification) as? CardknoxCardReaderCallback
        
        let code = callback?.code()
        
        let name = callback?.name()
        
        addLineToOutput(line: "Card reader - " + name!)
        
        if(code == CardknoxCardReaderCallbackType.error())
        {
            let errorMessage = callback?.message()
            addLineToOutput(line: "Card reader - " + errorMessage!)
        }
    }
    
    func customUIscanCompletedNotification(aNotification: Notification)
    {
        // Documentation purposes
        if(false){
            let scanCompleted = CardknoxSDKCustomUIBLEScanCompleted.from(aNotification) as! CardknoxSDKCustomUIBLEScanCompleted;
            let devices : Array<Any> = scanCompleted.scannedDevices();
            var scannedDevice: CardknoxSDKCustomUIBLEScannedDevice? = nil;
            
            // Example to get a first scanned VP3300 device.
            // All VP3300 devices have their internal name start with IDTECH prefix
            if(devices != nil && devices.count > 0)
            {
                for dev in devices{
                    let typedDevice = dev as! CardknoxSDKCustomUIBLEScannedDevice
                    let name = typedDevice.name()!;
                    
                    if name.hasPrefix("IDTECH")
                    {
                        // Save the reference & use it with the "connect to device" method
                        scannedDevice = typedDevice;
                        break;
                    }
                }
            }
        }
        
        let scanCompleted = CardknoxSDKCustomUIBLEScanCompleted.from(aNotification) as! CardknoxSDKCustomUIBLEScanCompleted
        scannedDevices = scanCompleted.scannedDevices() as! [CardknoxSDKCustomUIBLEScannedDevice]
 
        if(!scannedDevices.isEmpty)
        {
            scannedDevice = scannedDevices[0]
        }
    }
    
    // Define a function that accepts a Notification.
    // This method will be invoked when the SDK sends scanned device information
    func customUIscannedDeviceSubscription(aNotification: Notification)
    {
        // Documentation purposes
        if(false){
            // Use the SDK's "response" object utility method to transform a Notification into a "response" object
            scannedDevice = CardknoxSDKCustomUIBLEScannedDevice.from(aNotification) as! CardknoxSDKCustomUIBLEScannedDevice;
            let name = scannedDevice.name();
            let displayName = scannedDevice.displayName();
            let uuid = scannedDevice.uuid();
        }
        
        // Use the SDK's "response" object utility method to transform a Notification into a "response" object
        let device = CardknoxSDKCustomUIBLEScannedDevice.from(aNotification) as! CardknoxSDKCustomUIBLEScannedDevice
        addLineToOutput(line: "Scanned device: " + device.name() + " - " + device.displayName() + " - " + device.uuid());
    }
    
    func addLineToOutput(line : String)
    {
        NSLog("----- TEST ----- : %@", line);
        DispatchQueue.main.async
        {
            output = output + line + "\n"
        }
    }
    
    func startScanningButtinClick()
    {
        scannedDevice = nil
        scannedDevices = [CardknoxSDKCustomUIBLEScannedDevice]()
        
        let timeout = Int32(scanningTimeout ?? "") ?? 0
        cardknoxSDKManager.getCardknoxSDKCustomUIBluetooth()?.startScanning(withTimeout: timeout)
    }
    
    func stopScanningButtinClick()
    {
        cardknoxSDKManager.getCardknoxSDKCustomUIBluetooth()?.stopScanning()
    }
    
    func connectButtinClick()
    {
        // Documentation purposes
        if(false)
        {
            let exampleDevice : CardknoxSDKCustomUIBLEScannedDevice? = nil;
            let exampleCustomUI : CardknoxSDKCustomUIBLE? = nil;
            exampleCustomUI!.connect(withUUID: (exampleDevice!).uuid());
        }
                       
        if(scannedDevice != nil)
        {
            // Place a call to "connect to the device"
            // Expect appropriate card reader events in the NotificationCenter and act accordingly.
            cardknoxSDKManager.getCardknoxSDKCustomUIBluetooth().connect(withName: scannedDevice.name());
        }
    }
    
    func disconnectButtinClick()
    {
        cardknoxSDKManager.getCardknoxSDKCustomUIBluetooth()?.disconnectFromCurrentDevice()
    }
    
    func startTransactionButtonClick()
    {
        let prms : TransactionParameters = TransactionParameters.init()
        
        let command = commandType as NSString?
        prms.xCommand = command
        
        prms.vp3300TransactionTimeout = 13;
        
        let amountDouble = Double( amount.replacingOccurrences(of: ",", with: ".")) ?? 0
        prms.xAmount = amountDouble
        
        cardknoxSDKManager.getCardknoxSDKCustomUIBluetooth()?.startTransaction(withArgs: prms)
    }
    
    func stopTransactionButtinClick()
    {
        cardknoxSDKManager.getCardknoxSDKCustomUIBluetooth()?.cancelTransaction()
    }
}

