package com.cardknox.android.sdk.sampleapp2;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

import com.cardknox.android.sdk.sampleapp2.databinding.PaymentEngineFragmentBinding;

import cardknox.payments.sdk.CardknoxPaymentEngineSDK;
import cardknox.payments.sdk.CardknoxPaymentEngineSDKRequest;
import cardknox.payments.sdk.CardknoxPaymentEngineSDKSettings;
import cardknox.payments.sdk.CardknoxSDK;
import cardknox.payments.sdk.PaymentTransactionResponse;
import cardknox.payments.sdk.CardknoxPaymentEngineSDKRequest;
import cardknox.payments.sdk.CardknoxPaymentEngineSDKSettings;

public class PaymentEngineFragment extends Fragment{

    private PaymentEngineFragmentBinding binding;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        // Configure the Cardknox SDK with an xKey
        CardknoxSDK.create().setPrincipalKey(MainActivity2.KEY);

        // Supply the Cardknox SDK with the required values:
        // xSoftware name: name of your application
        // xVersion: Cardknox Gateway version. Currently 4.5.9
        CardknoxSDK.create().setSoftwareConfigurations(
                // xSoftwareName
                "Cardknox Android SDK sample Java app - Payment Engine",
                // xVersion
                "4.5.9");
    }

    @Override
    public void onResume() {
        super.onResume();
        RegisterAllIntentFilters();
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState)
    {
        binding = PaymentEngineFragmentBinding.inflate(inflater, container, false);
        return binding.getRoot();
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        // Buttons
        binding.btnDeviceInitialize.setOnClickListener(view1 -> executeDeviceInitialize());
        binding.btnDeviceCancelTransaction.setOnClickListener(view1 -> executeDeviceCancelTransaction());
        binding.btnProcessOutOfScope.setOnClickListener(view1 -> executeProcessOutOfScope());
        binding.btnClearoutput.setOnClickListener(v -> binding.output.setText(""));

        boolean localDevice = true;
        if (localDevice) {
            binding.inputPort.setText("10009");
            binding.inputDeviceName.setText("Pax_A77.3");
            binding.inputIpaddress.setText("172.20.7.84");
            binding.inputTimeout.setText("60"); // seconds, later converted to milliseconds
        }
    }

    private void executeDeviceInitialize(){

        CardknoxPaymentEngineSDKSettings settings = this.LocalSettings();
        CardknoxSDK.create().getPaymentEngine().Device_Initialize(settings);
    }

    private void executeDeviceCancelTransaction(){
        CardknoxSDK.create().getPaymentEngine().Device_CancelTransaction();
    }

    private void executeProcessOutOfScope(){
        CardknoxPaymentEngineSDKSettings settings = this.LocalSettings();
        CardknoxPaymentEngineSDKRequest request = new CardknoxPaymentEngineSDKRequest();

        double amount;
        try {
            amount = Double.parseDouble(binding.inputAmount.getText().toString());
        }
        catch (Exception e) {
            amount = 0.0;
        }

        request.SetxCommand("cc:sale");
        request.SetxAmount(amount);
        request.SetxAllowDuplicate(true);
        CardknoxSDK.create().getPaymentEngine().ProcessOutOfScope(settings, request);
    }

    private CardknoxPaymentEngineSDKSettings LocalSettings() {
        CardknoxPaymentEngineSDKSettings settings = new CardknoxPaymentEngineSDKSettings();
        settings.SetDevice_IP_Port(binding.inputPort.getText().toString());
        settings.SetDevice_IP_Address(binding.inputIpaddress.getText().toString());
        settings.SetDevice_Name(binding.inputDeviceName.getText().toString());

        try {
            String timeoutString = binding.inputTimeout.getText().toString();
            int timeoutSeconds = Integer.getInteger(timeoutString);
            settings.SetDevice_Timeout(timeoutSeconds * 1000); // Convert to seconds
        } catch (Exception e) {
            settings.SetDevice_Timeout(120 * 1000); // 120 seconds;
        }

        return settings;
    }


    /**
     * This is called when switching Fragments.
     * Fragment will be removed from the screen.
     * Can't rely on onDetach - it is not invoked on switching Fragments
     *
     * Good place to unregister BroadcastReceivers.
     * */
    @Override
    public void onPause() {
        super.onPause();
        UnregisterAllIntentFilters();
    }

    private void RegisterAllIntentFilters()
    {
        Context c = getContext();

        if(c != null)
        {
            Utils.tryRegisterReceiver(c, Receiver_TransactionProcessing, new IntentFilter(CardknoxSDK.TRANSACTION_CALLBACK_INTENTFILTER_ACTION()));
            Utils.tryRegisterReceiver(c, Receiver_DeviceStatus, new IntentFilter(CardknoxPaymentEngineSDK.DEVICE_STATUS_INTENTFILTER_ACTION()));
            Utils.tryRegisterReceiver(c, Receiver_TransactionStatus, new IntentFilter(CardknoxPaymentEngineSDK.TRANSACTION_STATUS_INTENTFILTER_ACTION()));
        }
    }

    private void UnregisterAllIntentFilters()
    {
        Context c = this.getContext();

        if(c != null)
        {
            Utils.tryUnregisterReceiver(c, Receiver_TransactionProcessing);
            Utils.tryUnregisterReceiver(c, Receiver_DeviceStatus);
            Utils.tryUnregisterReceiver(c, Receiver_TransactionStatus);
        }
    }

    private final BroadcastReceiver Receiver_DeviceStatus = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();

            // Verify that the Cardknox SDK broadcasted the event
            if(action.equals(CardknoxPaymentEngineSDK.DEVICE_STATUS_INTENTFILTER_ACTION()))
            {
                String extraKey = CardknoxPaymentEngineSDK.DEVICE_STATUS_INTENTFILTER_EXTRA();
                String deviceStatus = intent.getStringExtra(extraKey);
                Output(deviceStatus);
            }
        }
    };

    private final BroadcastReceiver Receiver_TransactionStatus = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();

            // Verify that the Cardknox SDK broadcasted the event
            if(action.equals(CardknoxPaymentEngineSDK.TRANSACTION_STATUS_INTENTFILTER_ACTION()))
            {
                String extraKey = CardknoxPaymentEngineSDK.TRANSACTION_STATUS_INTENTFILTER_EXTRA();
                String deviceStatus = intent.getStringExtra(extraKey);
                Output(deviceStatus);
            }
        }
    };

    private final BroadcastReceiver Receiver_TransactionProcessing = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            String action = intent.getAction();

            // Verify that the Cardknox SDK broadcasted the Intent
            if(action.equals(CardknoxSDK.TRANSACTION_CALLBACK_INTENTFILTER_ACTION()))
            {
                String extraKey = CardknoxSDK.TRANSACTION_CALLBACK_INTENTFILTER_EXTRA();
                java.lang.Object extra = intent.getParcelableExtra(extraKey);

                // Cast the 'extra' to the Cardknox SDK response type
                PaymentTransactionResponse response = (PaymentTransactionResponse)extra;

                if(response.GetIsSuccess())
                {
                    Toast.makeText(getContext(), "Transaction approved.", Toast.LENGTH_SHORT).show();
                    Output("Transaction approved. Ref num: " + response.GetxRefNum());
                }
                else{
                    Toast.makeText(getContext(), "Transaction declined/errored", Toast.LENGTH_SHORT).show();
                    Output("Transaction declined/errored. Ref num: " + response.GetxRefNum());
                }
            }
        }
    };

    private void Output(String text) {
        if (text != null && text.length() > 0)
        {
            Utils.log(text);
            binding.output.append("\n" + text);
        }
    }
}
