package com.cardknox.android.sdk.sampleapp2;

import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ArrayAdapter;
import android.widget.Toast;

import androidx.activity.result.ActivityResultCallback;
import androidx.activity.result.ActivityResultLauncher;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

import com.cardknox.android.sdk.sampleapp2.databinding.CustomuiUsbFragmentBinding;
import com.cardknox.android.sdk.sampleapp2.options.TransactionParametersContract;

import cardknox.payments.sdk.CardknoxCardReaderCallback;
import cardknox.payments.sdk.CardknoxSDK;
import cardknox.payments.sdk.PaymentTransactionResponse;
import cardknox.payments.sdk.TransactionParameters;
import cardknox.payments.sdk.VP3300StartTransactionMethod;

public class CustomUIUSBFragment extends Fragment
        implements ActivityResultCallback<TransactionParameters>
{
    private static final String SUPPORTED_IDTECH_USB_DEVICE_NO_CHOICE = "Select a USB device...";
    private static final String DEVICE_VP3300_BT_USB = "VP3300_BT_USB (PID 13632)";
    private static final String DEVICE_VP3350_COMPOSITE = "VP3350 COMPOSITE (PID 18001)";
    private static String[] SUPPORTED_IDTECH_USB_DEVICES = new String[]{
            SUPPORTED_IDTECH_USB_DEVICE_NO_CHOICE,
            DEVICE_VP3300_BT_USB,
            DEVICE_VP3350_COMPOSITE
    };

    public static final String STARTTRANSACTIONMETHOD_CTLS = "STARTTRANSACTIONMETHOD_CTLS";
    public static final String STARTTRANSACTIONMETHOD_EMV = "STARTTRANSACTIONMETHOD_EMV";
    public static final String STARTTRANSACTIONMETHOD_MSR = "STARTTRANSACTIONMETHOD_MSR";
    public static final String STARTTRANSACTIONMETHOD_DEVICE = "STARTTRANSACTIONMETHOD_DEVICE";
    private static String[] SUPPORTED_START_TRANSACTION_METHODS = new String[]{
            STARTTRANSACTIONMETHOD_CTLS,
            STARTTRANSACTIONMETHOD_MSR,
            STARTTRANSACTIONMETHOD_EMV,
            STARTTRANSACTIONMETHOD_DEVICE
    };

    private CustomuiUsbFragmentBinding binding;
    private String[] commands;
    private TransactionParameters currentParameters = new TransactionParameters();

    private String getSelectedTransactionType() {
        return binding.spinnerTransactionType.getSelectedItem().toString();
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        // Configure the Cardknox SDK with an xKey
        CardknoxSDK.create().setPrincipalKey(MainActivity2.KEY);

        // Supply the Cardknox SDK with the required values:
        // xSoftware name: name of your application
        // xVersion: Cardknox Gateway version. Currently 4.5.9
        CardknoxSDK.create().setSoftwareConfigurations(
                // xSoftwareName
                "Cardknox Android SDK sample Java app - custom ui",
                // xVersion
                "4.5.9");
    }

    @Override
    public View onCreateView(
            LayoutInflater inflater, ViewGroup container,
            Bundle savedInstanceState
    ) {

        commands = Commands.GetCommandsArray();
        binding = CustomuiUsbFragmentBinding.inflate(inflater, container, false);
        return binding.getRoot();
    }

    /**
     * Called when the Fragment is about to become visible on the UI.
     * Good place to register BroadcastReceivers.
     * */
    @Override
    public void onAttach(@NonNull Context context) {
        super.onAttach(context);
        RegisterAllIntentFilters();
    }

    @Override
    public void onPause() {
        super.onPause();
        UnregisterAllIntentFilters();
    }

    @Override
    public void onResume() {
        super.onResume();
        RegisterAllIntentFilters();
    }

    public void onViewCreated(@NonNull View view, Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        // Command dropdown
        ArrayAdapter<String> aa = new ArrayAdapter<String>(getActivity(),
                R.layout.my_simple_spinner_item,
                commands);

        aa.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        binding.spinnerTransactionType.setAdapter(aa);

        // Timeouts spinner
        ArrayAdapter<String> idtechUsbDevicesAdapter = new ArrayAdapter<>(getActivity(),
                android.R.layout.simple_spinner_item,
                SUPPORTED_IDTECH_USB_DEVICES);
        idtechUsbDevicesAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        binding.spinnerIdtechUsbDevices.setAdapter(idtechUsbDevicesAdapter);

        // Start transaction method spinner
        ArrayAdapter<String> startTransactionMethodsAdapter = new ArrayAdapter<>(getActivity(),
                android.R.layout.simple_spinner_dropdown_item,
                SUPPORTED_START_TRANSACTION_METHODS);
        idtechUsbDevicesAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        binding.spinnerStarttransactionMethod.setAdapter(startTransactionMethodsAdapter);

        // Buttons
        binding.btnConnect.setOnClickListener(Button_ConnectListener);
        binding.btnDisconnect.setOnClickListener(Button_DisconnectListener);
        binding.btnStartTransaction.setOnClickListener(Button_StartTransaction);
        binding.btnStopTransaction.setOnClickListener(Button_StopTransaction);
        binding.btnClearoutput.setOnClickListener(v -> binding.output.setText(""));
        binding.btnMoreoptions.setOnClickListener(v -> openOptionsScreen());
    }

    private void openOptionsScreen(){
        openOptionsLauncher.launch(currentParameters);
    }

    private final ActivityResultLauncher<TransactionParameters> openOptionsLauncher =
            registerForActivityResult(new TransactionParametersContract(), this);

    @Override
    public void onActivityResult(TransactionParameters result)
    {
        // Snapshot current state that should not be changed
        String oldCommand = currentParameters.GetxCommand();
        double oldAmount = currentParameters.GetxAmount();

        // Apply all changes
        currentParameters = result;

        // Restore state for snapshoted values
        currentParameters.SetxCommand(oldCommand);
        currentParameters.SetxAmount(oldAmount);
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        binding = null;
    }

    private final View.OnClickListener Button_ConnectListener = new View.OnClickListener() {
        @Override
        public void onClick(View view) {

            Object a = binding.spinnerIdtechUsbDevices.getSelectedItem();
            String selectedDevice = a != null ? a.toString() : null;

            if(selectedDevice != null){
                switch(selectedDevice){
                    case DEVICE_VP3300_BT_USB:
                        CardknoxSDK.create().getCustomUIUSB().connectVP3300();
                        break;
                    case DEVICE_VP3350_COMPOSITE:
                        Toast.makeText(getActivity(), "VP3350 support coming soon!", Toast.LENGTH_SHORT).show();
                        break;
                    case SUPPORTED_IDTECH_USB_DEVICE_NO_CHOICE:
                        Toast.makeText(getActivity(), "Error: no devices selected...", Toast.LENGTH_SHORT).show();
                        break;
                    default:
                        throw new RuntimeException("connect: can't figure out device");
                }
            }
            else {
                Toast.makeText(getContext(), "Cannot connect. Select a device to connect to", Toast.LENGTH_SHORT).show();
            }
        }
    };

    private final View.OnClickListener Button_DisconnectListener = view -> CardknoxSDK.create().getCustomUIUSB().disconnectFromCurrentDevice();

    private final View.OnClickListener Button_StartTransaction = new View.OnClickListener() {
        @Override
        public void onClick(View view) {

            // Validate amount
            double amount;
            try {
                amount = Double.parseDouble(binding.amount.getText().toString());
            } catch (NumberFormatException e) {
                Toast.makeText(getContext(), "Amount is invalid", Toast.LENGTH_SHORT).show();
                return;
            }

            currentParameters.SetxCommand(getSelectedTransactionType());
            currentParameters.SetxAmount(amount);
            currentParameters.SetVP3300TransactionTimeout(16);

            Object o = binding.spinnerStarttransactionMethod.getSelectedItem();

            VP3300StartTransactionMethod method = null;
            if(o instanceof String){
                String m = (String)o;
                if(m.equals(STARTTRANSACTIONMETHOD_CTLS))
                    method = VP3300StartTransactionMethod.CTLS;
                else if(m.equals(STARTTRANSACTIONMETHOD_EMV))
                    method = VP3300StartTransactionMethod.EMV;
                else if(m.equals(STARTTRANSACTIONMETHOD_MSR))
                    method = VP3300StartTransactionMethod.MSR;
                else if(m.equals(STARTTRANSACTIONMETHOD_DEVICE))
                    method = VP3300StartTransactionMethod.DEVICE;
            }

            if(method == null){
                Toast.makeText(getActivity(), "No start transaction method selected", Toast.LENGTH_SHORT).show();
            }
            else{
                currentParameters.SetVP3300StartTransactionMethod(method);
                CardknoxSDK.create().getCustomUIUSB().startTransactionWithArgs(currentParameters);
            }
        }
    };

    private final View.OnClickListener Button_StopTransaction = new View.OnClickListener() {
        @Override
        public void onClick(View view) {
            try {
                CardknoxSDK.create().getCustomUIUSB().cancelTransaction();
            }
            catch(Exception e)
            {
                String message = "Stop transaction error: " + e.getMessage();
                Toast.makeText(getContext(), message, Toast.LENGTH_LONG).show();
            }
        }
    };

    private void RegisterAllIntentFilters()
    {
        Context c = getContext();

        if(c != null)
        {
            Utils.tryRegisterReceiver(c, Receiver_TransactionProcessing, new IntentFilter(CardknoxSDK.TRANSACTION_CALLBACK_INTENTFILTER_ACTION()));
            Utils.tryRegisterReceiver(c, Receiver_CardReaderEvents, new IntentFilter(CardknoxSDK.CARDREADER_CALLBACK_INTENTFILTER_ACTION()));
        }
    }

    private void UnregisterAllIntentFilters()
    {
        Context c = this.getContext();

        if(c != null)
        {
            Utils.tryUnregisterReceiver(c, Receiver_TransactionProcessing);
            Utils.tryUnregisterReceiver(c, Receiver_CardReaderEvents);
        }
    }

    private final BroadcastReceiver Receiver_TransactionProcessing = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            String action = intent.getAction();

            // Verify that the Cardknox SDK broadcasted the Intent
            if(action.equals(CardknoxSDK.TRANSACTION_CALLBACK_INTENTFILTER_ACTION()))
            {
                String extraKey = CardknoxSDK.TRANSACTION_CALLBACK_INTENTFILTER_EXTRA();
                java.lang.Object extra = intent.getParcelableExtra(extraKey);

                // Cast the 'extra' to the Cardknox SDK response type
                PaymentTransactionResponse response = (PaymentTransactionResponse)extra;

                if(response.GetIsSuccess())
                {
                    Toast.makeText(getContext(), "Transaction approved.", Toast.LENGTH_SHORT).show();
                    Output("Transaction approved. Ref num: " + response.GetxRefNum());
                }
                else{
                    Toast.makeText(getContext(), "Transaction declined/errored", Toast.LENGTH_SHORT).show();
                    Output("Transaction declined/errored. Ref num: " + response.GetxRefNum());
                }
            }
        }
    };

    private final BroadcastReceiver Receiver_CardReaderEvents = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            String action = intent.getAction();

            // Verify that the Cardknox SDK broadcasted the Intent
            if(action.equals(CardknoxSDK.CARDREADER_CALLBACK_INTENTFILTER_ACTION()))
            {
                String extraKey = CardknoxSDK.CARDREADER_CALLBACK_INTENTFILTER_EXTRA();
                CardknoxCardReaderCallback callback = intent.getParcelableExtra(extraKey);
                int code = callback.GetCode();
                String name = callback.GetName();
                String message = callback.GetMessage();
                String logTag = "CardknoxDroidSampleApp";

                Output("Card reader event: " + name);

                if(message != null && !message.isEmpty())
                {
                    Output("Card reader event: " + message);
                }
            }
        }
    };



    private void Output(String text) {
        if (text != null && text.length() > 0 && binding != null && binding.output != null)
            binding.output.append("\n" + text);
    }
}